<?php
/**
 * Plugin Name: Monitorio
 * Plugin URI: https://monitorio.eu/plugins
 * Description: Server monitoring integration with Monitorio.eu
 * Version: 1.0.1
 * Author: Monitorio
 * Author URI: https://monitorio.eu
 * License: GPL v2 or later
 * Text Domain: monitorio
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('MONITORIO_VERSION', '1.0.1');
define('MONITORIO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MONITORIO_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('MONITORIO_API_URL', 'https://api.monitorio.eu/plugin');
define('MONITORIO_LOGO_URL', 'https://monitorio.eu/assets/img/logo-monitorio.svg');

class MonitorioPlugin
{
    private $option_name = 'monitorio_settings';
    private $cron_key_option = 'monitorio_cron_key';
    
    /**
     * Constructor
     */
    public function __construct()
    {
        add_action('plugins_loaded', array($this, 'load_textdomain'));
        add_action('init', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Load plugin textdomain for translations
     */
    public function load_textdomain()
    {
        load_plugin_textdomain('monitorio', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /**
     * Initialize plugin
     */
    public function init()
    {
        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_styles'));
        
        // AJAX handlers
        add_action('wp_ajax_monitorio_validate', array($this, 'ajax_validate'));
        add_action('wp_ajax_monitorio_download', array($this, 'ajax_download'));
        add_action('wp_ajax_monitorio_check', array($this, 'ajax_system_check'));
        add_action('wp_ajax_monitorio_run', array($this, 'ajax_manual_run'));
        
        // Cron endpoints
        add_action('wp_ajax_nopriv_monitorio_cron', array($this, 'ajax_cron_handler'));
        add_action('wp_ajax_monitorio_cron', array($this, 'ajax_cron_handler'));
        
        // REST API
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }
    
    /**
     * Plugin activation
     */
    public function activate()
    {
        // Generate cron key
        if (!get_option($this->cron_key_option)) {
            update_option($this->cron_key_option, wp_generate_password(32, false));
        }
        
        // Create default directories
        $upload_dir = wp_upload_dir();
        $monitorio_dir = $upload_dir['basedir'] . '/monitorio';
        
        if (!file_exists($monitorio_dir)) {
            wp_mkdir_p($monitorio_dir);
        }
        
        // Set default options
        $defaults = array(
            'client_uuid' => '',
            'server_api_key' => '',
            'output_folder' => $monitorio_dir,
            'script_version' => '',
            'is_configured' => false,
        );
        
        if (!get_option($this->option_name)) {
            add_option($this->option_name, $defaults);
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate()
    {
        // Cleanup if needed
    }
    
    /**
     * Enqueue admin styles
     */
    public function enqueue_admin_styles($hook)
    {
        if (strpos($hook, 'monitorio') === false) {
            return;
        }
        
        wp_enqueue_style(
            'monitorio-admin',
            MONITORIO_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            MONITORIO_VERSION
        );
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu()
    {
        add_menu_page(
            'Monitorio',
            'Monitorio',
            'manage_options',
            'monitorio',
            array($this, 'admin_page'),
            'dashicons-chart-area',
            100
        );
    }
    
    /**
     * Initialize settings
     */
    public function settings_init()
    {
        register_setting('monitorio_settings', $this->option_name, array(
            'sanitize_callback' => array($this, 'sanitize_settings'),
        ));
    }
    
    /**
     * Sanitize settings
     */
    public function sanitize_settings($input)
    {
        $sanitized = array();
        
        $sanitized['client_uuid'] = isset($input['client_uuid']) 
            ? sanitize_text_field($input['client_uuid']) 
            : '';
            
        $sanitized['server_api_key'] = isset($input['server_api_key']) 
            ? sanitize_text_field($input['server_api_key']) 
            : '';
            
        $sanitized['output_folder'] = isset($input['output_folder']) 
            ? sanitize_text_field($input['output_folder']) 
            : '';
            
        // Preserve other settings
        $current = get_option($this->option_name);
        $sanitized['script_version'] = isset($current['script_version']) ? $current['script_version'] : '';
        $sanitized['is_configured'] = isset($current['is_configured']) ? $current['is_configured'] : false;
        
        return $sanitized;
    }
    
    /**
     * Admin page
     */
    public function admin_page()
    {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'monitorio'));
        }
        
        $options = get_option($this->option_name);
        $cron_key = get_option($this->cron_key_option);
        $is_configured = !empty($options['is_configured']);
        $last_run = get_option('monitorio_last_run', '');
        $last_check = get_option('monitorio_last_check', array());

        // Check if scripts exist
        $scripts_exist = file_exists(MONITORIO_PLUGIN_PATH . 'monitorio.php') 
            && file_exists(MONITORIO_PLUGIN_PATH . 'monitorio-check.php');
        
        ?>
        <style>
            :root {
                --monitorio-primary: #19b4f0;
                --monitorio-primary-dark: #0d9fd8;
                --monitorio-primary-light: #e8f7fc;
                --monitorio-success: #28a745;
                --monitorio-warning: #ffc107;
                --monitorio-danger: #dc3545;
                --monitorio-gray: #6c757d;
                --monitorio-light: #f8f9fa;
                --monitorio-dark: #343a40;
            }
            
            .monitorio-wrap {
                max-width: 1200px;
                margin: 20px 20px 20px 0;
            }
            
            .monitorio-header {
                border: 2px solid var(--monitorio-primary);
                color: white;
                padding: 30px;
                border-radius: 8px;
                margin-bottom: 20px;
                display: flex;
                align-items: center;
                gap: 20px;
            }
            
            .monitorio-header img {
                height: 50px;
            }
            
            .monitorio-header h1 {
                margin: 0;
                font-size: 28px;
                font-weight: 600;
            }
            
            .monitorio-header p {
                margin: 5px 0 0 0;
                opacity: 0.9;
            }
            
            .monitorio-cards {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
                gap: 20px;
                margin-bottom: 20px;
            }
            
            .monitorio-card {
                background: white;
                border-radius: 8px;
                box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                overflow: hidden;
            }
            
            .monitorio-card-header {
                background: var(--monitorio-light);
                padding: 15px 20px;
                border-bottom: 1px solid #e9ecef;
                display: flex;
                align-items: center;
                gap: 10px;
            }
            
            .monitorio-card-header h2 {
                margin: 0;
                font-size: 16px;
                font-weight: 600;
                color: var(--monitorio-dark);
            }
            
            .monitorio-card-header .dashicons {
                color: var(--monitorio-primary);
            }
            
            .monitorio-card-body {
                padding: 20px;
            }
            
            .monitorio-form-group {
                margin-bottom: 15px;
            }
            
            .monitorio-form-group label {
                display: block;
                font-weight: 600;
                margin-bottom: 5px;
                color: var(--monitorio-dark);
            }
            
            .monitorio-form-group input[type="text"],
            .monitorio-form-group input[type="password"] {
                width: 100%;
                padding: 10px 12px;
                border: 1px solid #ddd;
                border-radius: 4px;
                font-size: 14px;
            }
            
            .monitorio-form-group input:focus {
                border-color: var(--monitorio-primary);
                outline: none;
                box-shadow: 0 0 0 2px var(--monitorio-primary-light);
            }
            
            .monitorio-form-group .description {
                font-size: 12px;
                color: var(--monitorio-gray);
                margin-top: 5px;
            }
            
            .monitorio-btn {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                padding: 10px 20px;
                border: none;
                border-radius: 4px;
                font-size: 14px;
                font-weight: 500;
                cursor: pointer;
                text-decoration: none;
                transition: all 0.2s;
            }
            
            .monitorio-btn-primary {
                background: linear-gradient(135deg, var(--monitorio-primary-dark) 0%, var(--monitorio-primary) 100%);
                color: white;
            }
            
            .monitorio-btn-primary:hover {
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(25, 180, 240, 0.4);
                color: white;
            }
            
            .monitorio-btn-secondary {
                background: var(--monitorio-light);
                color: var(--monitorio-dark);
                border: 1px solid #ddd;
            }
            
            .monitorio-btn-secondary:hover {
                background: #e9ecef;
            }
            
            .monitorio-btn-success {
                background: var(--monitorio-success);
                color: white;
            }
            
            .monitorio-btn-warning {
                background: var(--monitorio-warning);
                color: var(--monitorio-dark);
            }
            
            .monitorio-btn-danger {
                background: var(--monitorio-danger);
                color: white;
            }
            
            .monitorio-btn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
            }
            
            .monitorio-status {
                display: flex;
                align-items: center;
                gap: 10px;
                padding: 15px;
                border-radius: 4px;
                margin-bottom: 15px;
            }
            
            .monitorio-status-success {
                background: #d4edda;
                color: #155724;
            }
            
            .monitorio-status-warning {
                background: #fff3cd;
                color: #856404;
            }
            
            .monitorio-status-error {
                background: #f8d7da;
                color: #721c24;
            }
            
            .monitorio-status-info {
                background: var(--monitorio-primary-light);
                color: var(--monitorio-primary-dark);
            }
            
            .monitorio-stats {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 15px;
            }
            
            .monitorio-stat {
                text-align: center;
                padding: 15px;
                background: var(--monitorio-light);
                border-radius: 4px;
            }
            
            .monitorio-stat-value {
                font-size: 24px;
                font-weight: 700;
                color: var(--monitorio-primary);
            }
            
            .monitorio-stat-label {
                font-size: 12px;
                color: var(--monitorio-gray);
                text-transform: uppercase;
            }
            
            .monitorio-code {
                background: var(--monitorio-dark);
                color: #f8f9fa;
                padding: 15px;
                border-radius: 4px;
                font-family: monospace;
                font-size: 12px;
                overflow-x: auto;
                white-space: pre-wrap;
                word-break: break-all;
            }
            
            .monitorio-check-results {
                max-height: 400px;
                overflow-y: auto;
            }
            
            .monitorio-check-item {
                display: flex;
                align-items: center;
                gap: 10px;
                padding: 8px 0;
                border-bottom: 1px solid #eee;
            }
            
            .monitorio-check-item:last-child {
                border-bottom: none;
            }
            
            .monitorio-check-icon {
                width: 20px;
                height: 20px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 12px;
                color: white;
            }
            
            .monitorio-check-icon.success { background: var(--monitorio-success); }
            .monitorio-check-icon.warning { background: var(--monitorio-warning); color: var(--monitorio-dark); }
            .monitorio-check-icon.error { background: var(--monitorio-danger); }
            
            .monitorio-spinner {
                display: inline-block;
                width: 16px;
                height: 16px;
                border: 2px solid #f3f3f3;
                border-top: 2px solid var(--monitorio-primary);
                border-radius: 50%;
                animation: monitorio-spin 1s linear infinite;
            }
            
            @keyframes monitorio-spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            
            .monitorio-buttons {
                display: flex;
                gap: 10px;
                flex-wrap: wrap;
            }
            
            #monitorio-ajax-message {
                margin-top: 15px;
            }
        </style>
        
        <div class="monitorio-wrap">
            <!-- Header -->
            <div class="monitorio-header">
                <img src="<?php echo esc_url(MONITORIO_LOGO_URL); ?>" alt="Monitorio">
                <div>
                    <h1><?php _e('Server Monitoring. Simplified', 'monitorio'); ?></h1>
                </div>
            </div>
            
            <div class="monitorio-cards">
                <!-- Configuration Card -->
                <div class="monitorio-card">
                    <div class="monitorio-card-header">
                        <span class="dashicons dashicons-admin-settings"></span>
                        <h2><?php _e('Configuration', 'monitorio'); ?></h2>
                    </div>
                    <div class="monitorio-card-body">
                        <?php if ($is_configured): ?>
                            <div class="monitorio-status monitorio-status-success">
                                <span class="dashicons dashicons-yes-alt"></span>
                                <?php _e('Plugin is configured and active', 'monitorio'); ?>
                            </div>
                        <?php else: ?>
                            <div class="monitorio-status monitorio-status-warning">
                                <span class="dashicons dashicons-warning"></span>
                                <?php _e('Plugin needs to be configured', 'monitorio'); ?>
                            </div>
                        <?php endif; ?>
                        
                        <form method="post" action="options.php" id="monitorio-settings-form">
                            <?php settings_fields('monitorio_settings'); ?>
                            
                            <div class="monitorio-form-group">
                                <label for="client_uuid"><?php _e('Client API Key', 'monitorio'); ?></label>
                                <input type="text" 
                                       id="client_uuid" 
                                       name="<?php echo $this->option_name; ?>[client_uuid]" 
                                       value="<?php echo esc_attr($options['client_uuid'] ?? ''); ?>"
                                       placeholder="xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx">
                                <p class="description"><?php _e('Your Client API Key from server configuration on Monitorio.eu', 'monitorio'); ?></p>
                            </div>
                            
                            <div class="monitorio-form-group">
                                <label for="server_api_key"><?php _e('Server API Key', 'monitorio'); ?></label>
                                <input type="text" 
                                       id="server_api_key" 
                                       name="<?php echo $this->option_name; ?>[server_api_key]" 
                                       value="<?php echo esc_attr($options['server_api_key'] ?? ''); ?>"
                                       placeholder="xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx">
                                <p class="description"><?php _e('Your Server API Key from server configuration on Monitorio.eu', 'monitorio'); ?></p>
                            </div>
                            
                            <div class="monitorio-form-group">
                                <label for="output_folder"><?php _e('Output Folder', 'monitorio'); ?></label>
                                <input type="text" 
                                       id="output_folder" 
                                       name="<?php echo $this->option_name; ?>[output_folder]" 
                                       value="<?php echo esc_attr($options['output_folder'] ?? ''); ?>">
                                <p class="description"><?php _e('Folder where monitorio.json will be saved', 'monitorio'); ?></p>
                            </div>
                            
                            <div class="monitorio-buttons">
                                <?php submit_button(__('Save Settings', 'monitorio'), 'monitorio-btn monitorio-btn-primary', 'submit', false); ?>
                                <button type="button" id="monitorio-validate-btn" class="monitorio-btn monitorio-btn-secondary">
                                    <span class="dashicons dashicons-yes"></span>
                                    <?php _e('Validate & Download', 'monitorio'); ?>
                                </button>
                            </div>
                        </form>
                        
                        <div id="monitorio-ajax-message"></div>
                    </div>
                </div>
                
                <!-- Status Card -->
                <div class="monitorio-card">
                    <div class="monitorio-card-header">
                        <span class="dashicons dashicons-chart-bar"></span>
                        <h2><?php _e('Status', 'monitorio'); ?></h2>
                    </div>
                    <div class="monitorio-card-body">
                        <div class="monitorio-stats">
                            <div class="monitorio-stat">
                                <div class="monitorio-stat-value">
                                    <?php echo $scripts_exist ? '✓' : '✗'; ?>
                                </div>
                                <div class="monitorio-stat-label"><?php _e('Scripts', 'monitorio'); ?></div>
                            </div>
                            <div class="monitorio-stat">
                                <div class="monitorio-stat-value">
                                    <?php echo esc_html($options['script_version'] ?? '-'); ?>
                                </div>
                                <div class="monitorio-stat-label"><?php _e('Version', 'monitorio'); ?></div>
                            </div>
                            <div class="monitorio-stat">
                                <div class="monitorio-stat-value" style="font-size: 14px;">
                                    <?php echo $last_run ? esc_html($last_run) : __('Never', 'monitorio'); ?>
                                </div>
                                <div class="monitorio-stat-label"><?php _e('Last Run', 'monitorio'); ?></div>
                            </div>
                        </div>
                        
                        <div class="monitorio-buttons" style="margin-top: 15px;">
                            <button type="button" id="monitorio-run-btn" class="monitorio-btn monitorio-btn-success" <?php echo !$scripts_exist ? 'disabled' : ''; ?>>
                                <span class="dashicons dashicons-controls-play"></span>
                                <?php _e('Run Now', 'monitorio'); ?>
                            </button>
                            <button type="button" id="monitorio-check-btn" class="monitorio-btn monitorio-btn-secondary" <?php echo !$scripts_exist ? 'disabled' : ''; ?>>
                                <span class="dashicons dashicons-search"></span>
                                <?php _e('System Check', 'monitorio'); ?>
                            </button>
                        </div>
                        
                        <div id="monitorio-run-message" style="margin-top: 15px;"></div>
                        <div id="monitorio-check-results" class="monitorio-check-results" style="margin-top: 15px; display: none;"></div>
                    </div>
                </div>

                <!-- Cron Setup Card -->
                <div class="monitorio-card">
                    <div class="monitorio-card-header">
                        <span class="dashicons dashicons-clock"></span>
                        <h2><?php _e('Cron Setup', 'monitorio'); ?></h2>
                    </div>
                    <div class="monitorio-card-body">
                        <div class="monitorio-status monitorio-status-info">
                            <span class="dashicons dashicons-info"></span>
                            <?php _e('You can enable automatic monitoring from Monitorio.eu cloud in your server settings, or set up one of the cron methods below.', 'monitorio'); ?>
                        </div>                        

                        <div class="monitorio-form-group">
                            <label><?php _e('AJAX Endpoint', 'monitorio'); ?></label>
                            <div class="monitorio-code">* * * * * curl -s "<?php echo admin_url('admin-ajax.php'); ?>?action=monitorio_cron&key=<?php echo esc_attr($cron_key); ?>" > /dev/null 2>&1</div>
                        </div>

                        <div class="monitorio-form-group">
                            <label><?php _e('REST API Endpoint', 'monitorio'); ?></label>
                            <div class="monitorio-code">* * * * * curl -s "<?php echo rest_url('monitorio/v1/run'); ?>?key=<?php echo esc_attr($cron_key); ?>" > /dev/null 2>&1</div>
                        </div>

                        <div class="monitorio-form-group">
                            <label><?php _e('Direct PHP', 'monitorio'); ?></label>
                            <div class="monitorio-code">* * * * * /usr/bin/php <?php echo MONITORIO_PLUGIN_PATH; ?>cron-runner.php <?php echo esc_attr($cron_key); ?></div>
                        </div>

                        <p><strong><?php _e('Security Key:', 'monitorio'); ?></strong> <code><?php echo esc_html($cron_key); ?></code></p>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
            var nonce = '<?php echo wp_create_nonce('monitorio_ajax'); ?>';
            
            // Validate & Download
            $('#monitorio-validate-btn').on('click', function() {
                var btn = $(this);
                var originalText = btn.html();
                btn.prop('disabled', true).html('<span class="monitorio-spinner"></span> <?php _e('Validating...', 'monitorio'); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'monitorio_download',
                        nonce: nonce,
                        client_uuid: $('#client_uuid').val(),
                        server_api_key: $('#server_api_key').val(),
                        output_folder: $('#output_folder').val()
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#monitorio-ajax-message').html(
                                '<div class="monitorio-status monitorio-status-success">' +
                                '<span class="dashicons dashicons-yes-alt"></span> ' +
                                response.data.message +
                                '</div>'
                            );
                            setTimeout(function() { location.reload(); }, 2000);
                        } else {
                            $('#monitorio-ajax-message').html(
                                '<div class="monitorio-status monitorio-status-error">' +
                                '<span class="dashicons dashicons-dismiss"></span> ' +
                                response.data.message +
                                '</div>'
                            );
                        }
                    },
                    error: function() {
                        $('#monitorio-ajax-message').html(
                            '<div class="monitorio-status monitorio-status-error">' +
                            '<span class="dashicons dashicons-dismiss"></span> <?php _e('Connection error', 'monitorio'); ?>' +
                            '</div>'
                        );
                    },
                    complete: function() {
                        btn.prop('disabled', false).html(originalText);
                    }
                });
            });
            
            // Manual Run
            $('#monitorio-run-btn').on('click', function() {
                var btn = $(this);
                var originalText = btn.html();
                btn.prop('disabled', true).html('<span class="monitorio-spinner"></span> <?php _e('Running...', 'monitorio'); ?>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'monitorio_run',
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#monitorio-run-message').html(
                                '<div class="monitorio-status monitorio-status-success">' +
                                '<span class="dashicons dashicons-yes-alt"></span> ' +
                                response.data.message +
                                '</div>'
                            );
                        } else {
                            $('#monitorio-run-message').html(
                                '<div class="monitorio-status monitorio-status-error">' +
                                '<span class="dashicons dashicons-dismiss"></span> ' +
                                response.data.message +
                                '</div>'
                            );
                        }
                    },
                    error: function() {
                        $('#monitorio-run-message').html(
                            '<div class="monitorio-status monitorio-status-error">' +
                            '<span class="dashicons dashicons-dismiss"></span> <?php _e('Error running monitoring', 'monitorio'); ?>' +
                            '</div>'
                        );
                    },
                    complete: function() {
                        btn.prop('disabled', false).html(originalText);
                    }
                });
            });
            
            // System Check
            $('#monitorio-check-btn').on('click', function() {
                var btn = $(this);
                var originalText = btn.html();
                btn.prop('disabled', true).html('<span class="monitorio-spinner"></span> <?php _e('Checking...', 'monitorio'); ?>');

                $('#monitorio-check-results').show().html('<p><span class="monitorio-spinner"></span> <?php _e('Running system check...', 'monitorio'); ?></p>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'monitorio_check',
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var summaryClass = response.data.summary.errors > 0 ? 'monitorio-status-error' :
                                (response.data.summary.warnings > 0 ? 'monitorio-status-warning' : 'monitorio-status-success');

                            var html = '<div class="monitorio-status ' + summaryClass + '" style="margin-bottom: 10px;">' +
                                '<strong><?php _e('Summary:', 'monitorio'); ?></strong> ' +
                                response.data.summary.errors + ' <?php _e('errors', 'monitorio'); ?>, ' +
                                response.data.summary.warnings + ' <?php _e('warnings', 'monitorio'); ?>' +
                                '</div>';

                            html += '<div style="max-height: 250px; overflow-y: auto; border: 1px solid #eee; border-radius: 4px; padding: 10px;">';
                            response.data.results.forEach(function(item) {
                                var iconClass = item.status === 'ok' ? 'success' : (item.status === 'warning' ? 'warning' : 'error');
                                var icon = item.status === 'ok' ? '✓' : (item.status === 'warning' ? '!' : '✗');
                                html += '<div class="monitorio-check-item">' +
                                    '<span class="monitorio-check-icon ' + iconClass + '">' + icon + '</span>' +
                                    '<span>' + item.message + '</span>' +
                                    '</div>';
                            });
                            html += '</div>';

                            $('#monitorio-check-results').html(html);
                        } else {
                            $('#monitorio-check-results').html(
                                '<div class="monitorio-status monitorio-status-error">' +
                                response.data.message +
                                '</div>'
                            );
                        }
                    },
                    error: function() {
                        $('#monitorio-check-results').html(
                            '<div class="monitorio-status monitorio-status-error">' +
                            '<?php _e('Error running system check', 'monitorio'); ?>' +
                            '</div>'
                        );
                    },
                    complete: function() {
                        btn.prop('disabled', false).html(originalText);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Validate and download scripts
     */
    public function ajax_download()
    {
        check_ajax_referer('monitorio_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'monitorio')));
        }
        
        $client_uuid = sanitize_text_field($_POST['client_uuid'] ?? '');
        $server_api_key = sanitize_text_field($_POST['server_api_key'] ?? '');
        $output_folder = sanitize_text_field($_POST['output_folder'] ?? '');
        
        if (empty($client_uuid) || empty($server_api_key)) {
            wp_send_json_error(array('message' => __('Client UUID and Server API Key are required', 'monitorio')));
        }
        
        // Call Monitorio API
        $response = wp_remote_get(MONITORIO_API_URL . '/download?' . http_build_query(array(
            'client_uuid' => $client_uuid,
            'server_api_key' => $server_api_key,
        )), array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection to Monitorio.eu failed', 'monitorio')));
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!isset($body['success']) || !$body['success']) {
            $error_msg = isset($body['message']) ? $body['message'] : __('Unknown error', 'monitorio');
            wp_send_json_error(array('message' => $error_msg));
        }
        
        // Save scripts
        $scripts = $body['data']['scripts'] ?? array();
        
        foreach ($scripts as $filename => $content) {
            $decoded = base64_decode($content);
            if ($decoded === false) {
                continue;
            }
            file_put_contents(MONITORIO_PLUGIN_PATH . $filename, $decoded);
        }
        
        // Save config
        $config = $body['data']['config'] ?? array();
        $config['folder'] = rtrim($output_folder, '/') . '/';
        file_put_contents(MONITORIO_PLUGIN_PATH . 'monitorio.cfg', json_encode($config, JSON_PRETTY_PRINT));

        // Extract version from downloaded monitorio.php
        $script_version = $this->extractScriptVersion();

        // Update options
        $options = array(
            'client_uuid' => $client_uuid,
            'server_api_key' => $server_api_key,
            'output_folder' => $output_folder,
            'script_version' => $script_version,
            'is_configured' => true,
        );

        // Force update by using wpdb directly
        global $wpdb;
        $serialized = maybe_serialize($options);
        $wpdb->update(
            $wpdb->options,
            array('option_value' => $serialized),
            array('option_name' => $this->option_name)
        );
        wp_cache_delete($this->option_name, 'options');
        wp_cache_delete('alloptions', 'options');
        
        // Create output folder if not exists
        if (!file_exists($output_folder)) {
            wp_mkdir_p($output_folder);
        }

        // Auto-run system check after download
        $check_result = $this->perform_system_check();

        // If system check passed (no errors), run first monitoring
        $monitoring_result = null;
        if ($check_result['errors'] === 0) {
            $monitoring_result = $this->run_monitoring();
        }

        wp_send_json_success(array(
            'message' => sprintf(
                __('Successfully downloaded scripts (v%s) and configuration', 'monitorio'),
                $script_version ?: '?'
            ),
            'system_check' => $check_result,
            'first_monitoring' => $monitoring_result !== null ? ($monitoring_result ? 'success' : 'failed') : 'skipped',
        ));
    }
    
    /**
     * Perform system check (reusable method)
     * @return array Results with 'results', 'errors', 'warnings' keys
     */
    private function perform_system_check()
    {
        $config_file = MONITORIO_PLUGIN_PATH . 'monitorio.cfg';

        $results = array();
        $errors = 0;
        $warnings = 0;

        // PHP Version
        $phpVersion = phpversion();
        $phpMajor = (int) explode('.', $phpVersion)[0];
        if ($phpMajor >= 7) {
            $results[] = array('status' => 'ok', 'message' => "PHP version: {$phpVersion}");
        } else {
            $results[] = array('status' => 'error', 'message' => "PHP version: {$phpVersion} (minimum: 7.0)");
            $errors++;
        }

        // Extensions
        $extensions = array('json', 'openssl', 'curl');
        foreach ($extensions as $ext) {
            if (extension_loaded($ext)) {
                $results[] = array('status' => 'ok', 'message' => "Extension '{$ext}' loaded");
            } else {
                $results[] = array('status' => 'error', 'message' => "Extension '{$ext}' not loaded");
                $errors++;
            }
        }

        // Functions
        $disabled = explode(',', ini_get('disable_functions'));
        $disabled = array_map('trim', $disabled);
        $required_funcs = array('exec', 'shell_exec');
        foreach ($required_funcs as $func) {
            if (in_array($func, $disabled)) {
                $results[] = array('status' => 'error', 'message' => "Function '{$func}' is disabled");
                $errors++;
            } else {
                $results[] = array('status' => 'ok', 'message' => "Function '{$func}' available");
            }
        }

        // Config file
        if (file_exists($config_file)) {
            $results[] = array('status' => 'ok', 'message' => 'Config file exists');

            $config = json_decode(file_get_contents($config_file), true);
            if ($config) {
                $results[] = array('status' => 'ok', 'message' => 'Config file is valid JSON');

                if (!empty($config['folder'])) {
                    if (is_dir($config['folder']) && is_writable($config['folder'])) {
                        $results[] = array('status' => 'ok', 'message' => "Output folder writable");
                    } else {
                        $results[] = array('status' => 'error', 'message' => "Output folder not writable");
                        $errors++;
                    }
                }

                if (!empty($config['password'])) {
                    $results[] = array('status' => 'ok', 'message' => 'Encryption password is set');
                } else {
                    $results[] = array('status' => 'warning', 'message' => 'No encryption password set');
                    $warnings++;
                }

                $tests_count = isset($config['tests']) ? count($config['tests']) : 0;
                $results[] = array('status' => 'ok', 'message' => "Tests defined: {$tests_count}");
            } else {
                $results[] = array('status' => 'error', 'message' => 'Config file is not valid JSON');
                $errors++;
            }
        } else {
            $results[] = array('status' => 'warning', 'message' => 'Config file not found');
            $warnings++;
        }

        // Scripts
        if (file_exists(MONITORIO_PLUGIN_PATH . 'monitorio.php')) {
            $results[] = array('status' => 'ok', 'message' => 'monitorio.php exists');
        } else {
            $results[] = array('status' => 'error', 'message' => 'monitorio.php not found');
            $errors++;
        }

        return array(
            'results' => $results,
            'errors' => $errors,
            'warnings' => $warnings,
        );
    }

    /**
     * AJAX: Run system check
     */
    public function ajax_system_check()
    {
        check_ajax_referer('monitorio_ajax', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'monitorio')));
        }

        $check_script = MONITORIO_PLUGIN_PATH . 'monitorio-check.php';

        if (!file_exists($check_script)) {
            wp_send_json_error(array('message' => __('Check script not found. Please download scripts first.', 'monitorio')));
        }

        $check_result = $this->perform_system_check();

        // Save results
        update_option('monitorio_last_check', array(
            'time' => current_time('mysql'),
            'errors' => $check_result['errors'],
            'warnings' => $check_result['warnings'],
        ));

        wp_send_json_success(array(
            'results' => $check_result['results'],
            'summary' => array(
                'errors' => $check_result['errors'],
                'warnings' => $check_result['warnings'],
            ),
        ));
    }
    
    /**
     * AJAX: Manual run
     */
    public function ajax_manual_run()
    {
        check_ajax_referer('monitorio_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied', 'monitorio')));
        }
        
        $result = $this->run_monitoring();
        
        if ($result) {
            wp_send_json_success(array('message' => __('Monitoring completed successfully', 'monitorio')));
        } else {
            wp_send_json_error(array('message' => __('Monitoring failed. Check error log.', 'monitorio')));
        }
    }
    
    /**
     * AJAX: Cron handler
     */
    public function ajax_cron_handler()
    {
        $key = isset($_GET['key']) ? sanitize_text_field($_GET['key']) : '';
        
        if ($key !== get_option($this->cron_key_option)) {
            wp_die('Unauthorized', 'Unauthorized', array('response' => 401));
        }
        
        $result = $this->run_monitoring();
        
        if ($result) {
            wp_die('Success');
        } else {
            wp_die('Error', 'Error', array('response' => 500));
        }
    }
    
    /**
     * Register REST routes
     */
    public function register_rest_routes()
    {
        // Endpoint pre lokálny cron (s key parametrom)
        register_rest_route('monitorio/v1', '/run', array(
            'methods' => 'GET',
            'callback' => array($this, 'rest_run_handler'),
            'permission_callback' => array($this, 'rest_permission_check'),
        ));

        // Endpoint pre externé volanie z Monitorio servera
        register_rest_route('monitorio/v1', '/trigger', array(
            'methods' => array('GET', 'POST'),
            'callback' => array($this, 'rest_trigger_handler'),
            'permission_callback' => array($this, 'rest_trigger_permission_check'),
        ));
    }

    /**
     * REST: Trigger handler - pre volanie z Monitorio servera
     */
    public function rest_trigger_handler($request)
    {
        $result = $this->run_monitoring();

        $tz = new \DateTimeZone('Europe/Bratislava');
        $now = new \DateTime('now', $tz);
        $time = $now->format('Y-m-d H:i:s');

        if ($result) {
            return new WP_REST_Response(array(
                'status' => 'ok',
                'message' => 'Monitoring completed successfully',
                'time' => $time,
            ), 200);
        } else {
            return new WP_REST_Response(array(
                'status' => 'error',
                'message' => 'Monitoring failed. Scripts may not be configured.',
                'time' => $time,
            ), 500);
        }
    }

    /**
     * REST: Trigger permission check - overuje X-Monitorio-Token hlavičku
     */
    public function rest_trigger_permission_check($request)
    {
        // Získanie tokenu z hlavičky
        $token = $request->get_header('X-Monitorio-Token');

        // Ak nie je v hlavičke, skúsime GET/POST parameter
        if (empty($token)) {
            $token = $request->get_param('token');
        }

        if (empty($token)) {
            return new WP_Error(
                'missing_token',
                'Missing X-Monitorio-Token header or token parameter',
                array('status' => 401)
            );
        }

        // Overenie proti uloženému server_api_key
        $options = get_option($this->option_name);
        $server_api_key = isset($options['server_api_key']) ? $options['server_api_key'] : '';

        if (empty($server_api_key)) {
            return new WP_Error(
                'not_configured',
                'Plugin is not configured',
                array('status' => 503)
            );
        }

        if ($token !== $server_api_key) {
            return new WP_Error(
                'invalid_token',
                'Invalid token',
                array('status' => 403)
            );
        }

        return true;
    }
    
    /**
     * REST: Run handler
     */
    public function rest_run_handler($request)
    {
        $result = $this->run_monitoring();
        
        if ($result) {
            return new WP_REST_Response('Success', 200);
        } else {
            return new WP_REST_Response('Error', 500);
        }
    }
    
    /**
     * REST: Permission check
     */
    public function rest_permission_check($request)
    {
        $key = $request->get_param('key');
        return $key === get_option($this->cron_key_option);
    }
    
    /**
     * Extract version from downloaded monitorio.php script
     */
    private function extractScriptVersion()
    {
        $script_path = MONITORIO_PLUGIN_PATH . 'monitorio.php';

        if (!file_exists($script_path)) {
            return '';
        }

        $content = file_get_contents($script_path);

        // Look for: private $version = '0.4.9';
        if (preg_match('/private\s+\$version\s*=\s*[\'"]([^\'"]+)[\'"]/', $content, $matches)) {
            return $matches[1];
        }

        return '';
    }

    /**
     * Run monitoring script
     */
    private function run_monitoring()
    {
        $script_path = MONITORIO_PLUGIN_PATH . 'monitorio.php';
        $config_path = MONITORIO_PLUGIN_PATH . 'monitorio.cfg';
        
        if (!file_exists($script_path) || !file_exists($config_path)) {
            error_log('Monitorio: Scripts or config not found');
            return false;
        }
        
        try {
            // Include monitorio script
            include_once($script_path);
            
            if (class_exists('Monitorio')) {
                $monitorio = new Monitorio($config_path);
                $monitorio->testServer();
            }
            
            // Update stats
            $tz = new \DateTimeZone('Europe/Bratislava');
            $now = new \DateTime('now', $tz);
            update_option('monitorio_last_run', $now->format('Y-m-d H:i:s'));
            
            return true;
        } catch (\Exception $e) {
            error_log('Monitorio error: ' . $e->getMessage());
            return false;
        }
    }
}

// Initialize plugin
new MonitorioPlugin();